<?php

namespace App\Http\Controllers;

use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CategoryController extends Controller
{
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer une catégorie.');
        }
        $validated = $request->validate([
            'name' => ['required', 'string', 'min:2', 'unique:categories,name'],
        ]);
        $category = Category::create([
            'name' => $validated['name'],
            'slug' => Category::generateUniqueSlug(),
        ]);
        return redirect()->route('categories.edit', $category)->with('success', 'Catégorie créée avec succès.');
    }

    public function update(Request $request, Category $category)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas modifier une catégorie.');
        }
        $validated = $request->validate([
            'name' => ['required', 'string', 'min:2'],
            'description' => ['nullable', 'string'],
        ]);
        if ($validated['name'] !== $category->name) {
            $validated['slug'] = Category::generateUniqueSlug($validated['name']);
        }
        $category->update($validated);
        return redirect()->route('categories.show', $category->refresh())->with('success', 'Catégorie modifiée avec succès.');
    }

    public function destroy(Category $category)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer une catégorie.');
        }
        if ($category->products()->count() > 0) {
            return back()->with('error', 'Vous ne pouvez pas supprimer une catégorie avec des produits.');
        }
        $category->delete();
        return redirect()->route('categories.index')->with('success', 'Catégorie supprimée avec succès.');
    }
}
