<?php

namespace App\Http\Controllers;

use App\Models\SaleType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SaleTypeController extends Controller
{
    public function store(Request $request)
    {

        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer un type de vente.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'min:2'],
        ]);

        $saleType = SaleType::create([
            'name' => $validated['name'],
            'code' => SaleType::generateUniqueCode($validated['name']),
            'status' => SaleType::STATUS_ACTIVE,
            'sort_order' => SaleType::max('sort_order') + 1,
        ]);

        return redirect()->route('sale-types.edit', $saleType)->with('success', 'Type de vente créé avec succès.');
    }

    public function update(Request $request, SaleType $saleType)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas modifier un type de vente.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'min:2'],
        ]);

        $saleType->update([
            'name' => $validated['name'],
        ]);

        return redirect()->route('sale-types.edit', $saleType)->with('success', 'Type de vente modifié avec succès.');
    }

    public function activate(SaleType $saleType)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas activer un type de vente.');
        }

        $saleType->update(['status' => SaleType::STATUS_ACTIVE]);
        return back()->with('success', 'Type de vente activé avec succès.');
    }

    public function deactivate(SaleType $saleType)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas désactiver un type de vente.');
        }

        $saleType->update(['status' => SaleType::STATUS_INACTIVE]);
        return back()->with('success', 'Type de vente désactivé avec succès.');
    }

    public function destroy(SaleType $saleType)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer un type de vente.');
        }

        if ($saleType->sales->count() > 0) {
            return back()->with('error', 'Vous ne pouvez pas supprimer un type de vente avec des ventes.');
        }

        $saleType->delete();

        return redirect()->route('sale-types.index')->with('success', 'Type de vente supprimé avec succès.');
    }
}
