<?php

namespace App\Http\Controllers;

use App\Models\Address;
use App\Models\Contact;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SupplierController extends Controller
{
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer un fournisseur.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
        ]);

        if (Supplier::where('name', $validated['name'])->exists()) {
            return redirect()->route('suppliers.index')
                ->with('warning', 'Un fournisseur avec ce nom existe déjà.');
        }

        $supplier = Supplier::create([
            'name' => $validated['name'],
            'slug' => method_exists(Supplier::class, 'generateSlug') ? Supplier::generateSlug() : str()->slug($validated['name']).'-'.uniqid(),
            'status' => false,
        ]);

        $contact = new Contact;
        $contact->save();
        $supplier->contact_id = $contact->id;

        $address = new Address;
        $address->save();
        $supplier->address_id = $address->id;

        $supplier->save();

        return redirect()->route('suppliers.edit', $supplier)
            ->with('success', 'Fournisseur créé avec succès.');
    }

    public function update(Request $request, Supplier $supplier)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas modifier un fournisseur.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'contact_person' => ['nullable', 'string', 'max:255'],
        ]);

        $supplier->update([
            'name' => $validated['name'],
            'contact_person' => $validated['contact_person'] ?? $supplier->contact_person,
        ]);

        return back()->with('success', 'Fournisseur modifié avec succès.');
    }

    public function activate(Supplier $supplier)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas activer un fournisseur.');
        }

        $hasAllRequiredFields = $supplier->hasAllRequiredFields();
        if (! $hasAllRequiredFields['success']) {
            return back()->with('error', $hasAllRequiredFields['message']);
        }

        if ($supplier->contact === null) {
            return back()->with('error', 'Le fournisseur doit avoir une personne de contact.');
        }

        if ($supplier->contact->phone === null) {
            return back()->with('error', 'Le fournisseur doit avoir un numéro de téléphone.');
        }

        if ($supplier->address === null) {
            return back()->with('error', 'Le fournisseur doit avoir une adresse.');
        }

        if ($supplier->address->province_id === null) {
            return back()->with('error', 'Le fournisseur doit avoir une province.');
        }

        if ($supplier->address->city_id === null) {
            return back()->with('error', 'Le fournisseur doit avoir une ville.');
        }

        $supplier->update(['status' => true]);

        return redirect()->route('suppliers.show', $supplier)->with('success', 'Fournisseur activé avec succès.');
    }

    public function deactivate(Supplier $supplier)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas désactiver un fournisseur.');
        }

        $supplier->update(['status' => false]);

        return back()->with('success', 'Fournisseur désactivé avec succès.');
    }

    public function destroy(Supplier $supplier)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer un fournisseur.');
        }

        if ($supplier->purchaseOrders()->count() > 0) {
            return back()->with('error', 'Vous ne pouvez pas supprimer un fournisseur avec des bons de commande.');
        }

        $supplier->delete();

        return redirect()->route('suppliers.index')->with('success', 'Fournisseur supprimé avec succès.');
    }
}
