<?php

namespace App\Http\Controllers;

use App\Models\Unit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class UnitController extends Controller
{
    public function store(Request $request)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas créer une unité.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'min:1', 'unique:units,name'],
            'symbol' => ['required', 'string', 'max:10', 'unique:units,symbol'],
        ]);

        $unit = Unit::create([
            'name' => $validated['name'],
            'symbol' => $validated['symbol'],
            'slug' => Unit::generateUniqueSlug($validated['name']),
        ]);

        return redirect()->route('units.edit', $unit)->with('success', 'Unité créée avec succès.');
    }

    public function update(Request $request, Unit $unit)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas modifier une unité.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'min:1', 'unique:units,name,' . $unit->id],
            'symbol' => ['required', 'string', 'max:10', 'unique:units,symbol,' . $unit->id],
        ]);

        $unit->update([
            'name' => $validated['name'],
            'symbol' => $validated['symbol'],
        ]);

        return redirect()->route('units.edit', $unit)->with('success', 'Unité modifiée avec succès.');
    }

    public function destroy(Unit $unit)
    {
        if (Auth::user()->role !== 'owner' && Auth::user()->role !== 'admin') {
            return back()->with('error', 'Vous ne pouvez pas supprimer une unité.');
        }

        // Check if any products use this unit as purchase or sale unit
        $productsUsingUnit = $unit->productsAsPurchaseUnit()->count() + $unit->productsAsSaleUnit()->count();
        
        if ($productsUsingUnit > 0) {
            return back()->with('error', "Vous ne pouvez pas supprimer cette unité car {$productsUsingUnit} produit(s) l'utilisent encore.");
        }

        $unit->delete();
        
        return redirect()->route('units.index')->with('success', 'Unité supprimée avec succès.');
    }
}
