<?php

namespace App\Models;

use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\Auth;

class Admin extends Authenticatable
{
    use HasFactory;

    const ROLE_SUPER_ADMIN = 'super_admin';
    const ROLE_ADMIN = 'admin';

    protected $guarded = [];

    protected $casts = [
        'last_activity_at' => 'datetime',
    ];

    /**
     * Get the user that owns the admin.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the role for this admin.
     */
    public function role()
    {
        return $this->belongsTo(Role::class);
    }

    /**
     * Check if admin has a specific permission.
     */
    public function hasPermission($permission)
    {
        if ($this->is_super_admin) {
            return true;
        }

        if ($this->permissions && in_array($permission, $this->permissions)) {
            return true;
        }

        return false;
    }

    /**
     * Check if admin can manage merchants.
     */
    public function canManageMerchants()
    {
        return $this->is_super_admin || $this->can_manage_merchants;
    }

    /**
     * Check if admin can manage users.
     */
    public function canManageUsers()
    {
        return $this->is_super_admin || $this->can_manage_users;
    }

    /**
     * Check if admin can manage products.
     */
    public function canManageProducts()
    {
        return $this->is_super_admin || $this->can_manage_products;
    }

    /**
     * Check if admin can manage orders.
     */
    public function canManageOrders()
    {
        return $this->is_super_admin || $this->can_manage_orders;
    }

    /**
     * Check if admin can view reports.
     */
    public function canViewReports()
    {
        return $this->is_super_admin || $this->can_view_reports;
    }

    /**
     * Check if admin can manage settings.
     */
    public function canManageSettings()
    {
        return $this->is_super_admin || $this->can_manage_settings;
    }

    /**
     * Update last activity.
     */
    public function updateLastActivity()
    {
        $this->update([
            'last_activity_at' => now(),
            'last_activity_ip' => request()->ip(),
        ]);
    }

    /**
     * Get full name of admin.
     */
    public function getFullNameAttribute()
    {
        return $this->user ? $this->user->firstname . ' ' . $this->user->lastname : 'Unknown';
    }

    /**
     * Get email of admin.
     */
    public function getEmailAttribute()
    {
        return $this->user ? $this->user->email : null;
    }

    /**
     * Get phone of admin.
     */
    public function getPhoneAttribute()
    {
        return $this->user ? $this->user->phone : null;
    }

    /**
     * Scope to get active admins only.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to get super admins only.
     */
    public function scopeSuperAdmins($query)
    {
        return $query->where('is_super_admin', true);
    }

    /**
     * Check if current user is an admin.
     */
    public static function isCurrentUserAdmin()
    {
        $user = Auth::user();
        if (!$user) {
            return false;
        }

        return self::where('user_id', $user->id)
                   ->where('is_active', true)
                   ->exists();
    }

    /**
     * Get current admin instance.
     */
    public static function getCurrentAdmin()
    {
        $user = Auth::user();
        if (!$user) {
            return null;
        }

        return self::where('user_id', $user->id)
                   ->where('is_active', true)
                   ->first();
    }
} 