<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Customer extends Model
{
    use \Illuminate\Database\Eloquent\SoftDeletes;
    protected $guarded = [];

    const STATUS_ACTIVE = 1;
    const STATUS_INACTIVE = 0;

    const CUSTOMER_STATUS = [
        ['id' => 1, 'name' => 'Actif', 'slug' => 'active'],
        ['id' => 0, 'name' => 'Inactif', 'slug' => 'inactive'],
    ];

    public function getRouteKeyName()
    {
        return 'slug';
    }

    public function merchant(): BelongsTo
    {
        return $this->belongsTo(Merchant::class);
    }

    public function address(): BelongsTo
    {
        return $this->belongsTo(Address::class);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function sales(): HasMany
    {
        return $this->hasMany(Sale::class);
    }

    public function debts(): HasMany
    {
        return $this->hasMany(Sale::class)->where('sale_type_id', Sale::PAYMENT_METHOD_DEBT);
    }

    public function debtRecords(): HasMany
    {
        return $this->hasMany(Debt::class);
    }

    /**
     * Get the total debt amount for this customer.
     */
    public function getTotalDebtAmountAttribute(): float
    {
        return $this->debtRecords()->sum('remaining_amount');
    }

    /**
     * Get the total paid amount for this customer.
     */
    public function getTotalPaidAmountAttribute(): float
    {
        return $this->debtRecords()->sum('paid_amount');
    }

    /**
     * Get the formatted total debt amount.
     */
    public function getFormattedTotalDebtAmountAttribute(): string
    {
        return number_format($this->total_debt_amount, 0, ',', ' ') . ' ' . \App\Models\Setting::DEFAULT_CURRENCY;
    }

    /**
     * Check if customer has any pending debts.
     */
    public function hasPendingDebts(): bool
    {
        return $this->debtRecords()->where('status', '!=', Debt::STATUS_PAID)->exists();
    }

    /**
     * Get pending debts for this customer.
     */
    public function getPendingDebts()
    {
        return $this->debtRecords()->where('status', '!=', Debt::STATUS_PAID)->get();
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', true);
    }

    public function scopeInactive($query)
    {
        return $query->where('status', false);
    }

    // Accessors
    public function getFullNameAttribute()
    {
        return $this->name;
    }

    public function isActive(): bool
    {
        return $this->status === self::STATUS_ACTIVE;
    }

    public function getStatusLabelAttribute(): string
    {
        return $this->isActive() ? 'Actif' : 'Inactif';
    }

    public function getStatusBadge()
    {
        return match ($this->status) {
            self::STATUS_ACTIVE => '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-green-100 text-green-800 min-w-30 justify-center"><i class="ph ph-check-circle mr-1"></i>Actif</span>',
            default => '<span class="inline-flex items-center px-2.5 py-0.5 rounded text-xs font-medium bg-red-100 text-red-800 min-w-30 justify-center"><i class="ph ph-x-circle mr-1"></i>Inactif</span>',
        };
    }
}
