<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class Expense extends Model
{
    use HasFactory;

    protected $guarded = [];

    const STATUS_DRAFT = false;  // boolean false = draft

    const STATUS_APPROVED = true;  // boolean true = approved

    protected $casts = [
        'amount' => 'decimal:2',
        'expense_date' => 'date',
        'status' => 'boolean',
    ];

    public const STATUS_LABELS = [
        ['id' => self::STATUS_DRAFT, 'name' => 'Brouillon', 'slug' => 'draft'],
        ['id' => self::STATUS_APPROVED, 'name' => 'Approuvé', 'slug' => 'approved'],
    ];

    public function getRouteKeyName()
    {
        return 'slug';
    }

    protected static function boot()
    {
        parent::boot();
        self::creating(function ($model) {
            $model->slug = self::generateUniqueSlug();
            $model->status = self::STATUS_DRAFT;
        });
    }

    // Relationships
    public function category(): BelongsTo
    {
        return $this->belongsTo(ExpenseCategory::class, 'expense_category_id');
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    // Scopes
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeByCategory($query, $categoryId)
    {
        return $query->where('expense_category_id', $categoryId);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('expense_date', [$startDate, $endDate]);
    }

    // Methods
    public function isDraft(): bool
    {
        return $this->status === self::STATUS_DRAFT;
    }

    public function isApproved(): bool
    {
        return $this->status === self::STATUS_APPROVED;
    }

    public function canBeApproved(): bool
    {
        return $this->isDraft();
    }

    public function canBeRejected(): bool
    {
        return $this->isDraft();
    }

    public function canBeMarkedAsPaid(): bool
    {
        return $this->isApproved();
    }

    public function getStatusLabelAttribute()
    {
        return $this->status == self::STATUS_APPROVED ? 'Approuvé' : 'Brouillon';
    }

    public function getStatusBadgeAttribute()
    {
        if ($this->status == self::STATUS_APPROVED) {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded min-w-30 text-xs font-medium bg-green-100 text-green-800 border border-green-500 justify-center">
                <i class="ph ph-check-circle mr-1"></i>
                Approuvé
            </span>';
        } else {
            return '<span class="inline-flex items-center px-2.5 py-0.5 rounded min-w-30 text-xs font-medium bg-gray-100 text-gray-800 border border-gray-500 justify-center">
                <i class="ph ph-x-circle mr-1"></i>
                Brouillon
            </span>';
        }
    }

    public function getFormattedAmountAttribute(): string
    {
        return number_format($this->amount, 2, ',', ' ').' CDF';
    }

    public function getFormattedDateAttribute(): string
    {
        return $this->expense_date->format('d/m/Y');
    }

    public static function generateUniqueSlug()
    {
        return Str::orderedUuid();
    }
}
